/// @file deviceDetector.cpp
///
/// @brief Device detection main class
///
/// This file implements the DeviceDetector class and the enum it uses.
/// The aim of this class is to provide a common interface for USPI components
/// to detect devices' events and get information from them.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see DeviceInfo DeviceDetector Udev

#include "deviceDetector.h"
#include "udev.h"
#include "eventHandler.h"

#define LOGGER_IDENTITY "DeviceDetector" ///< Identity to be used by the logger

#include "logger.h"
#include "syslogLogger.h"
#include "dltLogger.h"

namespace adit {
namespace uspi {

/// @brief Object constructor
/// @param logType The type of log to initialize.
///
/// It can accept 0 or 1 parameter. By default the object will initialize
/// its internal logger with @ref DD_LOG_DLT. The user can anyway choose to
/// override this behavior.
/// On creation, the Udev instance will be created and initialized, the
/// device list will therefore be populated and ready to be checked against
/// the device @ref mMask.
DeviceDetector::DeviceDetector(IDeviceEventHandler& deviceHandler,
                               enum DD_LOG_TYPE logType):
    mMask(0UL),
    mStandAlone(false),
    mDeviceHandler(deviceHandler)
{
    switch (logType) {
    case DD_LOG_DLT:
        ddlog.rdbuf(new DltLog());
        break;
    case DD_LOG_COUT:
        ddlog.rdbuf(std::cout.rdbuf());
        break;
    case DD_LOG_CLOG:
        ddlog.rdbuf(std::clog.rdbuf());
        break;
    case DD_LOG_NONE:
        ddlog.rdbuf(nullptr);
        break;
    default:
        // Default to syslog
        break;
    };

    Udev& u = Udev::getInstance();
    u.setDeviceDetector(this);
}

/// @brief Object destructor
///
/// In stand alone mode, the @ref EventHandler loop termination will be
/// triggered, and the object will wait for the thread to be stopped.
/// The logger is also destroyed, no log will be handled from this point on.
DeviceDetector::~DeviceDetector()
{
    if (mStandAlone)
        (EventHandler::getInstance()).terminateEventLoop(true);

    ddlog.rdbuf(nullptr);
}

/// @brief Checks if a device must generate a notification.
/// @param dInfo The @ref DeviceInfo to check
/// @param eType The event type received
///
/// This method is executed by the Udev class once a new device has been
/// added or removed from the device list.
void DeviceDetector::checkDevice(DeviceInfo& dInfo, enum DD_EVENT_TYPE eType)
{
    ddlog << kLogDebug << "Checking " << dInfo.mMask << " for notification."
        << std::endl;

    if (dInfo.mMask & mMask) {
        mDevInfo = dInfo;
        mDeviceHandler.handleDeviceEvent(mDevInfo, eType, dInfo.mMask & mMask);
    }
}

/// @brief Starts the stand alone mode.
///
/// Not thread safe. The EventHandler thread will be created and the object
/// @ref mStandAlone will be set to true. Once this method is executed, the
/// @ref getEventFd() and @ref dispatchEvent() throw an error if executed.
void DeviceDetector::setStandAlone(void)
{
    EventHandler& eh = EventHandler::getInstance();
    Udev& u = Udev::getInstance();

    if (mStandAlone)
        return;

    mStandAlone = true;

    u.setStandAlone();
    eh.startEventLoop();
}

/// @brief Dispatches the event occurring on the @ref getEventFd() file
/// descriptor.
///
/// This method shall be executed by the client if and only if the library
/// is not put in stand alone mode using the @ref setStandAlone method.
/// Throw an error if standalone has been set to true
int DeviceDetector::dispatchEvent()
{
    if (mStandAlone)
        throw std::runtime_error("Device detection started as stand alone");

    return (Udev::getInstance()).dispatchEvent();
}

/// @brief Returns the file descriptor for polling.
///
/// Throw an error if standalone has been set to true.
int DeviceDetector::getEventFd()
{
    if (mStandAlone)
        throw std::runtime_error("Device detection started as stand alone");

    return (Udev::getInstance()).getFd();
}

/// @brief Sets the @ref mMask value.
/// @param inMask The mask to set.
///
/// This will trigger the check of the device list. Any newly watched device
/// will generate an event, and thus call to @ref handleDeviceEvent() will
/// be performed.
void DeviceDetector::setMask(uint32_t inMask)
{
    Udev& u = Udev::getInstance();

    // First notify for devices that where there but not watched
    mMask = inMask & ~mMask;
    u.checkAll();

    // Now set the new mask for future events;
    mMask = inMask;
}

} // namespace uspi
} // namespace adit
